package de.hdm_stuttgart.mi.sd1.task2;

import java.util.Arrays;

import static de.hdm_stuttgart.mi.sd1.task2.PathValidity.*;

/**
 * <p>Representing mobile phone unlock patterns.</p>
 */
public class UnlockPattern {

    /**
     * <p>Creating a swipe path.</p>
     *
     * <p>Mobile phone unlocking swipe paths are being represented by indexing a phone unlock screen's nine possible
     *    node positions from 0 to 8:</p>
     *
     * <table class="goikTableDefaults">
     *     <caption>Mobile phone swipe pattern representation example</caption>
     *     <tr>
     *         <th colspan="2">A phone's display</th>
     *         <th style="width:30ex">Path parameterization</th>
     *         <th>Code representation</th>
     *     </tr>
     *     <tr>
     *
     *        <td>
     *           <img src="doc-files/unlock.svg" alt="Unlocking a phone by swiping" width="100">
     *        </td>
     *
     *        <td>
     *           <pre style="padding: 1ex;border: 1px solid black;"
     *             ><span class="myGreen">0 ← 1 ← 2</span>
     *<!--       --><span class="myGreen">↓</span>
     *<!--       --><span class="myGreen">3   4</span>   5
     *<!--       --><span class="myGreen">↓ ↗</span>
     *<!--       --><span class="myGreen">6</span>   7   8</pre>
     *        </td>
     *
     *        <td>A pattern along the nodes
     *            {<code class="myGreen">2, 1, 0, 3, 6, 4</code>}.</td>
     *        <td>
     *            <pre><code class="java"
     *              >UnlockPattern path =
     *<!--        -->   new UnlockPattern(
     *<!--        -->      new byte[]{2, 1, 0, 3, 6, 4});</code></pre>
     *        </td>
     *     </tr>
     * </table>
     *
     * @param swipePath A series of nodes forming an unlocking pattern. Altering the argument array's values after
     *                  creation does not affect the instance.
     */
    public UnlockPattern(final byte[] swipePath) {
        this.swipePath = Arrays.copyOf(swipePath, swipePath.length);
    }

    /**
     * <p>Unlock path equality is defined as node by node equality.</p>
     *
     * @param other A second swipe path.
     *
     * @return <code>true</code> if both paths are equal, <code>false</code> otherwise.
     *
     * <br>
     *
     * <section class="implementationHints">
     *    <h4>Hints:</h4>
     *    <p><a
     *     href="https://freedocs.mi.hdm-stuttgart.de/__slidesw1ChapterInheritance.html#/sda_inherit_fig_ShapeEquals"
     *     target="_blank"
     *       >How to define <b>equals(...)</b> example</a>.</p>
     * </section>
     */
    @Override
    public boolean equals(final Object other) {
        if (other instanceof final UnlockPattern pattern) {
            return Arrays.equals(swipePath, pattern.swipePath);
        } else {
            return false;
        }
    }

    /**
     * <p>A swipe path's validity.</p>
     *
     * <p>Each node in a swipe path can only reach its <em>immediate</em> neighbors: We can move exactly one step right,
     *    left, up, down or diagonally. Consider the following sample:</p>
     *
     * <table class="goikTableDefaults">
     *     <caption>An example of <span class="myGreen">reachable</span> and <span class="myRed">unreachable</span>
     *              node neighbors</caption>
     *     <tr>
     *
     *        <td>
     *           <pre style="padding: 1ex;display: inline-block;border: 1px solid black;"
     *             > 0 <span class="myGreen">🢂 1</span>   <span class="myRed">2</span>
     * <!--        --> <span class="myGreen">🢃 🢆</span>
     * <!--        --> <span class="myGreen">3</span>   <span class="myGreen">4</span>   <span class="myRed">5</span>
     * <!--        -->
     * <!--        --> <span class="myRed"  >6</span>   <span class="myRed"  >7</span>   <span class="myRed">8</span></pre>
     *        </td>
     *
     *        <td style="width: 40em"> Starting from the top left position 0 we can reach <code class="myGreen">1</code>,
     *          <code class="myGreen">3</code> or <code class="myGreen">4</code> in one step. On contrary
     *         <code class="myRed">2</code>, <code class="myRed">5</code>, <code class="myRed">6</code>,
     *          <code class="myRed">7</code> or <code class="myRed">8</code> require at least two steps.</td>
     *
     *     </tr>
     * </table>
     *
     * <p>A path can be valid or invalid being reflected by its {@link PathValidity} value:</p>
     *
     * <table class="goikTableDefaults">
     *    <caption>Legal and illegal path samples</caption>
     *    <tr>
     *       <td style="width: 15%;word-wrap: break-word;"
     *           >{@link PathValidity#VALID}:<br>
     *            A valid path <span class="myGreen">{0, 3, 6, 7}</span></td>
     *       <td>
     *          <pre style="padding: 1ex;display: inline-block;border: 1px solid black;"
     *            > <span class="myGreen">0</span>   1   2
     * <!--     --> <span class="myGreen">🢃</span>
     * <!--     --> <span class="myGreen">3</span>   4   5
     * <!--     --> <span class="myGreen">🢃</span>
     * <!--     --> <span class="myGreen">6 🢂 7</span>   8</pre>
     *       </td>
     *    </tr>
     *    <tr>
     *       <td style="width: 15%;word-wrap: break-word;"
     *         >{@link PathValidity#GAP}:<br>
     *         An invalid path {<span class="myGreen">0, 3</span><span class="bgRed">,</span>
     *         <span class="myGreen">2, 5</span>} containing a gap between 3 and 2.</td>
     *       <td>
     *          <pre style="padding: 1ex;display: inline-block;border: 1px solid black;"
     *            > <span class="myGreen">0</span>   <span                                             >1</span>   <span class="myGreen">2</span>
     * <!--     --> <span class="myGreen">🢃   </span><span class="myRed">🢅</span><!--                                                   -->   <span class="myGreen">🢃</span>
     * <!--     --> <span class="myGreen">3</span>   <span                                             >4</span>   <span class="myGreen">5</span>
     * <!--     -->
     * <!--     --> <span                >6</span>   <span                                             >7</span>   <span                >8</span></pre>
     *       </td>
     *    </tr>
     *    <tr>
     *       <td style="width: 15%;word-wrap: break-word;"
     *          >{@link PathValidity#DUPLICATE}:<br>
     *           An invalid path {<span class="myGreen">0</span>, <span class="myRed">3</span>, <span class="myRed">3</span>,
     *           <span class="myGreen">4</span>} containing two duplicate neighboring nodes.</td>
     *       <td colspan="2">
     *          <p>Two immediate neighboring nodes must differ.</p>
     *       </td>
     *    </tr>
     *    <tr>
     *       <td style="width: 15%;word-wrap: break-word;"
     *          >{@link PathValidity#EMPTY}:<br>
     *           An invalid path {} not containing any nodes.</td>
     *       <td colspan="2">
     *          <p>A path must contain at least one node.</p>
     *       </td>
     *    </tr>
     * </table>
     *
     * <p>In case of multiple errors the first one with respect to the given node sequence is being returned.</p>
     *
     * @return Either path being {@link PathValidity#VALID} or a value describing the reason for being invalid.
     */
    public PathValidity isValid() {

        if (0 == swipePath.length) {
            return EMPTY;
        } else {
            int previous = swipePath[0];
            for (int i = 1; i < swipePath.length; i++) {
                final int current = swipePath[i];
                final PathValidity neighborValidity = isValidNeighbor[previous][current];
                if (VALID != neighborValidity) {
                    return neighborValidity;
                }
                previous = current;
            }
        }
        return VALID;
    }

    /**
     * <p>Get a path's string representation.</p>
     *
     * <p>Example:</p>
     *
     * <pre style="padding: 1ex;border: 1px solid black;"
     *       ><span class="myGreen">0 ← 1 ← 2</span>
     *<!-- --><span class="myGreen">↓</span>
     *<!-- --><span class="myGreen">3   4</span>   5
     *<!-- --><span class="myGreen">↓ ↗</span>
     *<!-- --><span class="myGreen">6</span>   7   8</pre>
     *
     * returns The path's nodes by index sequence e.g., <code>"{2,1,0,3,6,4}"</code> for the above path example.
     */
    @Override
    public String toString() {
        if (0 == swipePath.length) {
            return "{}";
        } else {
            final StringBuilder builder = new StringBuilder("{").append(Integer.toString(swipePath[0]));
            for (int i = 1; i < swipePath.length; i++) {
                builder.append(",").append(swipePath[i]);
            }
            return builder.append('}').toString();
        }
    }

    private final byte[] swipePath;

    static private final PathValidity[][] isValidNeighbor = {

            /*  Upper left corner  */ { DUPLICATE, VALID,     GAP,
            /**/                        VALID,     VALID,     GAP,
            /**/                        GAP,       GAP,       GAP       },

            /*  Upper center       */ { VALID,     DUPLICATE, VALID,
            /**/                        VALID,     VALID,     VALID,
            /**/                        GAP,       GAP,       GAP       },

            /*  Upper right corner */ { GAP,       VALID,     DUPLICATE,
            /**/                        GAP,       VALID,     VALID,
            /**/                        GAP,       GAP,       GAP       },

            /*  Middle left        */ { VALID,     VALID,     GAP,
            /**/                        DUPLICATE, VALID,     GAP,
            /**/                        VALID,     VALID,     GAP       },

            /*  Middle center      */ { VALID,     VALID,     VALID,
            /**/                        VALID,     DUPLICATE, VALID,
            /**/                        VALID,     VALID,     VALID     },

            /*  Middle right       */ { GAP,       VALID,     VALID,
            /**/                        GAP,       VALID,     DUPLICATE,
            /**/                        GAP,       VALID,     VALID     },

            /*  Lower left corner  */ { GAP,       GAP,       GAP,
            /**/                        VALID,     VALID,     GAP,
            /**/                        DUPLICATE, VALID,     GAP       },

            /*  Lower center       */ { GAP,       GAP,       GAP,
            /**/                        VALID,     VALID,     VALID,
            /**/                        VALID,     DUPLICATE, VALID     },

            /*  Lower right corner */ { GAP,       GAP,       GAP,
            /**/                        GAP,       VALID,     VALID,
            /**/                        GAP,       VALID,     DUPLICATE }
    };
}
